<?	// copyright (C) 2005 - Matt Newberry, matt@mattnewberry.net
	// all rights reserved, except as licensed under contract
	
	/*  A TextBlock is a data structure containing one or more paragraghs of text; the opening words of the 
		first paragraph may optionally be stored as a 'headline', which can then be displayed in contrasting html
		style, such as bold or italicized.
	
		Public Interface:
		
				//-- constructor ---------------
			
			function TextBlock($filename);
				- creates a TextBlock from an XML datafile; $filename should be a full path 
					relative to the executing script directory
			
				//-- display functions ---------------
			
			function printHeadline();
				- print the TextBlock's headline, html-encoded, the first time it is called; thereafter prints nothing.
			
			function next();
				- advance the paragraph pointer to the next paragraph; returns false if no more paragraphs
			
			function printNextParagraph();
				- print the current paragraph, html-encoded
			
				//-- error handling ---------------
			
			function parseFailed();
				- returns false if the TextBlock was successfully parsed from xml, otherwise true (from XmlSimpleBaseClass)
			
			function getParseError();
				- returns error code from the PHP xml parser; 0 if no error, 8001 if input file not found 
					or not readable (from XmlSimpleBaseClass)
			
			function getParseMessage();
				- returns displayable error message from the PHP xml parser (from XmlSimpleBaseClass)
			
				//-- admin functions ---------------
			
			function setHeadline($headline);
				- set the headline text for the TextBlock
			
			function addParagraph($paragraph);
				- add a paragraph of text to the TextBlock
			
			function addParagraphs($text);
				- add one or more paragraphs of text, separated by a blank line, to the TextBlock
			
			function save();
				- save data back to disk under original filename (from XmlSimpleBaseClass)
			
			function clear();
				- clear the headline and paragraphs
		
		
		Typical usage in HTML display context:
		
			$textblock = new TextBlock("data/textblock.xml");
			if (! $textblock->parseFailed()) {
				while ($textblock->next()) {
					$textblock->printHeadline();
					$textblock->printNextParagraph();
				}
			}
		
		
		XML Data Format:
		
			<textblock>
				<headline>TEXT</headline>
				<paragraphs>
					<paragraph>TEXT</paragraph>
							.
							.
				</paragraphs>
			</textblock>
	*/
	
	require_once("XmlSimpleBaseClass.class.php");
	
	class TextBlock extends XmlSimpleBaseClass {
		var $pm_headline = "";
		var $pm_paragraphs = array();
		var $_index = -1;
		var $_count = 0;
		var $_next = 0;
		
		function TextBlock($filename) {
			$this->XmlSimpleBaseClass($filename);
			$this->_count = count($this->pm_paragraphs);
		}
		
		function _p_start_element($parser, $element, &$attributes) {
			parent::_p_start_element($parser, $element, &$attributes);
			if ($this->_tag == "paragraph") {
				$this->pm_paragraphs[] = "";
				++$this->_index;
			}
		}
		
		function _p_cdata($parser, $text) {
			if ($this->_tag == "headline")
				$this->pm_headline .= $text;
			else if ($this->_tag == "paragraph")
				$this->pm_paragraphs[$this->_index] .= $text;
		}
		
		function clear() {
			$this->pm_headline = "";
			$this->pm_paragraphs = array();
			$this->_count = 0;
			$this->_next = 0;
		}
		
		function setHeadline($headline) {
			$this->pm_headline = $headline;
		}
		
		function addParagraph($paragraph) {
			$this->pm_paragraphs = $paragraph;
		}
		
		function addParagraphs($text) {
			while (($index = strpos($text, "\r\n\r\n")) > 0) {
				$this->pm_paragraphs[] = trim(substr($text, 0, $index));
				$text = trim(substr($text, $index+2));
			}
			$this->pm_paragraphs[] = $text;
			$this->_count = count($this->pm_paragraphs);
		}
		
		var $_headline_sent = false;
		
		function printHeadline() {
			if (! $this->_headline_sent) {
				print "<B><H2>";
				print htmlspecialchars(trim($this->pm_headline));
				print "</B></H2>";
				
			}
			$this->_headline_sent = true;
		}
		
		function next() {
			if ($this->_next < $this->_count)
				return true;
			else {
				$this->_next = 0;
				$this->pm_headline_sent = false;
				return false;
			}
		}
				
		function printNextParagraph() {
			print htmlspecialchars(trim($this->pm_paragraphs[$this->_next++]));
			print "<P>";
		}

		function getHeadline() {
			return $this->pm_headline;
		}

		function getParagraphs() {
			return $this->pm_paragraphs;
		}

	}
?>
